<?php
namespace TypechoPlugin\MoranAliyunOSS;

use Typecho\Plugin\PluginInterface;
use Typecho\Widget;
use Typecho\Widget\Helper\Form;
use Typecho\Widget\Helper\Form\Element\Text;
use Typecho\Widget\Helper\Form\Element\Password;
use Typecho\Widget\Helper\Form\Element\Select;
use Typecho\Date;
use Typecho\Common;
use Typecho\Widget\Helper\Form\Element\Textarea;
use Widget\Upload;

if (!defined('__TYPECHO_ROOT_DIR__')) exit;

/**
 * MoranAliyunOSS
 * Typecho 阿里云 OSS 上传插件
 *
 * @package MoranAliyunOSS
 * @author 默然
 * @version 1.0.0
 * @link https://www.moranblog.cn/
 */
class Plugin implements PluginInterface
{
    const UPLOAD_DIR = '/usr/uploads';

    /* ================= 生命周期 ================= */

    public static function activate()
    {
        \Typecho\Plugin::factory('Widget_Upload')->uploadHandle     = __CLASS__ . '::uploadHandle';
        \Typecho\Plugin::factory('Widget_Upload')->modifyHandle     = __CLASS__ . '::modifyHandle';
        \Typecho\Plugin::factory('Widget_Upload')->deleteHandle     = __CLASS__ . '::deleteHandle';
        \Typecho\Plugin::factory('Widget_Upload')->attachmentHandle = __CLASS__ . '::attachmentHandle';
        return _t('MoranAliyunOSS 启用成功');
    }

    public static function deactivate() {}

    /* ================= 配置页 ================= */

    public static function config(Form $form)
    {
        echo <<<HTML
<div style="margin:12px 0 16px; padding:12px 14px; border:1px solid #e5e7eb; border-left:4px solid #3b82f6; background:#f8fafc; color:#111827; border-radius:6px; line-height:1.75;">
    <div style="font-size:14px; font-weight:700; margin-bottom:8px;">MoranAliyunOSS 插件说明</div>
    <div style="font-size:13px; color:#374151; margin-bottom:8px;">
        本插件基于 <b>aliyun-oss-php-sdk-2.7.2</b> 实现，用于将 Typecho 附件上传到阿里云 OSS 并生成访问链接。
    </div>
    <ol style="margin:0; padding-left:18px; font-size:13px; color:#374151;">
        <li>插件更新于2025-01-29</li>
        <li>
            插件基于
            <a href="https://github.com/aliyun/aliyun-oss-php-sdk/releases/tag/v2.7.2" target="_blank" rel="noopener">aliyun-oss-php-sdk-2.7.2</a>
            开发，若以后 SDK 开发包更新导致插件不可用，请到
            <a href="https://www.moranblog.cn/plugins/aliyun-oss.html" target="_blank" rel="noopener">我的博客</a>
            获取新版本插件。<br/>
        </li>
        <li>如有问题或建议请到<a href="https://www.moranblog.cn/" target="_blank" rel="noopener">我的博客</a>留言。</li>
    </ol>
</div>
HTML;

        $ak = new Text('accessKey', null, '', _t('AccessKey ID'), _t('在阿里云控制台创建 AccessKey 后获取。'));
        $ak->addRule('required', _t('AccessKey ID 不能为空'));
        $form->addInput($ak);

        $sk = new Password('secretKey', null, '', _t('AccessKey Secret'), _t('请妥善保管，不要泄露。'));
        $sk->addRule('required', _t('AccessKey Secret 不能为空'));
        $form->addInput($sk);

        $endpoint = new Text('endpoint', null, '', _t('Endpoint'), _t('例如：oss-cn-hangzhou.aliyuncs.com（不要带 http/https）。'));
        $endpoint->addRule('required', _t('Endpoint 不能为空'));
        $form->addInput($endpoint);

        $bucket = new Text('bucket', null, '', _t('Bucket'), _t('Bucket 名称，例如：my-bucket'));
        $bucket->addRule('required', _t('Bucket 不能为空'));
        $form->addInput($bucket);

        $domain = new Text('domain', null, '', _t('访问域名（可选）'), _t('例如：https://cdn.example.com 或 https://bucket.endpoint/（建议带 https）。'));
        $form->addInput($domain);
    }

    public static function personalConfig(Form $form) {}

    /* ================= 上传 ================= */

    public static function uploadHandle($file)
    {
        if (empty($file['name'])) return false;

        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if (!Upload::checkFileType($ext)) return false;

        $opt  = Widget::widget('Widget_Options')->plugin('MoranAliyunOSS');
        $date = new Date(Widget::widget('Widget_Options')->gmtTime);

        $dir  = self::UPLOAD_DIR . '/' . $date->year . '/' . $date->month . '/' . $date->day;
        $name = sprintf('%u', crc32(uniqid())) . '.' . $ext;
        $path = $dir . '/' . $name;

        self::ossClient()->uploadFile($opt->bucket, ltrim($path, '/'), $file['tmp_name']);

        return [
            'name' => $file['name'],
            'path' => $path,
            'size' => $file['size'],
                'type' => $ext,
            'mime' => Common::mimeContentType($file['tmp_name'])
        ];
    }

    public static function modifyHandle($content, $file)
    {
        return self::uploadHandle($file);
    }

    /* ================= 删除 ================= */

    public static function deleteHandle($content)
    {
        if (empty($content['attachment']->path)) return false;
        $opt = Widget::widget('Widget_Options')->plugin('MoranAliyunOSS');
        self::ossClient()->deleteObject($opt->bucket, ltrim($content['attachment']->path, '/'));
        return true;
    }

    /* ================= URL ================= */

    public static function attachmentHandle($content)
    {
        $path = is_array($content) ? $content['path'] : $content->path;
        return self::urlPrefix() . '/' . ltrim($path, '/');
    }

    private static function ossClient(): \OSS\OssClient
    {
        static $client;
        if ($client) {
            return $client;
        }

        require_once __DIR__ . '/aliyun-oss-php-sdk-2.7.2/autoload.php';

        $opt = Widget::widget('Widget_Options')->plugin('MoranAliyunOSS');
        $client = new \OSS\OssClient(
            $opt->accessKey,
            $opt->secretKey,
            $opt->endpoint
        );

        return $client;
    }

    private static function urlPrefix(): string
    {
        $opt = Widget::widget('Widget_Options')->plugin('MoranAliyunOSS');
        return rtrim((string) $opt->domain, '/');
    }
}
